/*:
 * @target MZ
 * @plugindesc Title fantasy particles (start-filled, pool->lift, LB->RU drift, gusts, multi-color). v2.1.3
 * @author HS
 *
 * @help
 * タイトル画面に「静かな幻想」系の光粒パーティクルを表示します。
 * - 左下→右上の斜め流れ（drift）
 * - 画面下に滞留（pool）→ ふわっと上昇（rise）
 * - 常風 + ときどき突風（gust）
 * - 複数色（白/青白/緑白/黄色白など）
 * - 永久ループ（上へ抜けたらpoolへ戻る / 右へ抜けたら回り込み）
 *
 * StartFilled=true の場合、起動直後から画面全体に粒子が居る見え方になります。
 *
 * 画面サイズへの追従:
 * Margin / PoolHeight / EdgeFadePx は「px」または「比率(0-1)」で指定できます。
 * 例) PoolHeight=0.30 → 画面高さの30%
 *
 * v2.1.3 変更点:
 * - 右側が薄くなる問題対策：pool復帰時のXを「左固定」ではなく「左寄り“バイアス分布”+一部均等分布」に変更
 *   （PoolLeftWidthRate は “左寄りの強さ” として効きます。値を上げるほど均等になります）
 * - デフォルト ColorPalette に「黄色白」系を追加
 * - v2.1.1/2 の安定化修正を維持（ParticleImage拡張子対応/MinMax正規化/Bitmap更新/terminate確実remove）
 *
 * @param Enabled
 * @text 有効
 * @type boolean
 * @default true
 *
 * @param Layer
 * @text レイヤー
 * @type select
 * @option Below Foreground (logoの後ろ)
 * @value belowForeground
 * @option Above Foreground (logoの前/ウィンドウの後ろ)
 * @value aboveForeground
 * @option Above All (最前面)
 * @value aboveAll
 * @default aboveForeground
 *
 * @param ParticleCount
 * @text 粒数
 * @type number
 * @min 1
 * @max 300
 * @default 80
 *
 * @param StartFilled
 * @text 起動直後に全域へ分布
 * @type boolean
 * @default true
 *
 * @param StartPoolRatio
 * @text 起動直後pool比率(0-1)
 * @type number
 * @decimals 2
 * @min 0
 * @max 1
 * @default 0.25
 *
 * @param Margin
 * @text 画面外マージン(px or 0-1)
 * @type number
 * @decimals 2
 * @min 0
 * @default 160
 *
 * @param PoolHeight
 * @text 下部滞留ゾーン高さ(px or 0-1)
 * @type number
 * @decimals 2
 * @min 0
 * @default 0.30
 *
 * @param PoolLeftWidthRate
 * @text 溜まりの左寄り強さ(0-1) ※上げるほど均等
 * @type number
 * @decimals 2
 * @min 0
 * @max 1
 * @default 0.38
 *
 * @param ReleaseMinFrames
 * @text 滞留時間最小(frame)
 * @type number
 * @min 1
 * @default 120
 *
 * @param ReleaseMaxFrames
 * @text 滞留時間最大(frame)
 * @type number
 * @min 1
 * @default 360
 *
 * @param ReleaseFadeFrames
 * @text 上昇開始フェード(frame)
 * @type number
 * @min 0
 * @default 40
 *
 * @param ParticleImage
 * @text パーティクル画像(img/pictures)
 * @type file
 * @dir img/pictures
 * @default
 *
 * @param BaseTextureSize
 * @text 自動生成テクスチャサイズ(px)
 * @type number
 * @min 16
 * @max 256
 * @default 64
 *
 * @param BlendMode
 * @text 合成モード
 * @type select
 * @option SCREEN (静か)
 * @value SCREEN
 * @option ADD (強め発光)
 * @value ADD
 * @option NORMAL
 * @value NORMAL
 * @default SCREEN
 *
 * @param SizeMin
 * @text サイズ最小
 * @type number
 * @decimals 2
 * @default 0.16
 *
 * @param SizeMax
 * @text サイズ最大
 * @type number
 * @decimals 2
 * @default 0.52
 *
 * @param AlphaMin
 * @text 透明度最小
 * @type number
 * @decimals 2
 * @default 0.18
 *
 * @param AlphaMax
 * @text 透明度最大
 * @type number
 * @decimals 2
 * @default 0.60
 *
 * @param ColorPalette
 * @text 色パレット(#RRGGBBを,区切り)
 * @type string
 * @default #ffffff,#d8f0ff,#cfe8ff,#d8ffe8,#d4fff4,#fff2cc,#ffe5a8
 *
 * @param ColorJitter
 * @text 色ゆらぎ強さ(0-0.5)
 * @type number
 * @decimals 2
 * @min 0
 * @max 0.5
 * @default 0.10
 *
 * @param DriftXMin
 * @text 基本ドリフトX最小(px/frame)
 * @type number
 * @decimals 3
 * @default 0.06
 *
 * @param DriftXMax
 * @text 基本ドリフトX最大(px/frame)
 * @type number
 * @decimals 3
 * @default 0.22
 *
 * @param RiseYMin
 * @text 上昇Y最小(px/frame) ※負なら上へ
 * @type number
 * @decimals 3
 * @default -0.55
 *
 * @param RiseYMax
 * @text 上昇Y最大(px/frame) ※負なら上へ
 * @type number
 * @decimals 3
 * @default -0.18
 *
 * @param PoolBobAmpMin
 * @text 滞留ゆらぎ幅最小(px)
 * @type number
 * @decimals 1
 * @default 0.6
 *
 * @param PoolBobAmpMax
 * @text 滞留ゆらぎ幅最大(px)
 * @type number
 * @decimals 1
 * @default 2.8
 *
 * @param SwayAmpMin
 * @text 揺れ幅最小(px)
 * @type number
 * @decimals 1
 * @default 0.6
 *
 * @param SwayAmpMax
 * @text 揺れ幅最大(px)
 * @type number
 * @decimals 1
 * @default 3.4
 *
 * @param SwayFreqMin
 * @text 揺れ周波数最小
 * @type number
 * @decimals 3
 * @default 0.010
 *
 * @param SwayFreqMax
 * @text 揺れ周波数最大
 * @type number
 * @decimals 3
 * @default 0.035
 *
 * @param RotationSpeedMin
 * @text 回転速度最小(rad/frame)
 * @type number
 * @decimals 4
 * @default -0.010
 *
 * @param RotationSpeedMax
 * @text 回転速度最大(rad/frame)
 * @type number
 * @decimals 4
 * @default 0.010
 *
 * @param WindXMin
 * @text 常風X最小(px/frame)
 * @type number
 * @decimals 3
 * @default -0.06
 *
 * @param WindXMax
 * @text 常風X最大(px/frame)
 * @type number
 * @decimals 3
 * @default 0.10
 *
 * @param WindYMin
 * @text 常風Y最小(px/frame) ※負なら上へ
 * @type number
 * @decimals 3
 * @default -0.04
 *
 * @param WindYMax
 * @text 常風Y最大(px/frame) ※負なら上へ
 * @type number
 * @decimals 3
 * @default 0.02
 *
 * @param WindChangeFrames
 * @text 常風の変化周期(frame)
 * @type number
 * @min 1
 * @default 420
 *
 * @param GustChancePerSecond
 * @text 突風発生率(回/秒)
 * @type number
 * @decimals 3
 * @min 0
 * @default 0.020
 *
 * @param GustDurationMin
 * @text 突風時間最小(frame)
 * @type number
 * @min 1
 * @default 90
 *
 * @param GustDurationMax
 * @text 突風時間最大(frame)
 * @type number
 * @min 1
 * @default 180
 *
 * @param GustXMin
 * @text 突風X最小(px/frame)
 * @type number
 * @decimals 3
 * @default 0.25
 *
 * @param GustXMax
 * @text 突風X最大(px/frame)
 * @type number
 * @decimals 3
 * @default 0.55
 *
 * @param GustYMin
 * @text 突風Y最小(px/frame) ※負なら上へ
 * @type number
 * @decimals 3
 * @default -0.20
 *
 * @param GustYMax
 * @text 突風Y最大(px/frame) ※負なら上へ
 * @type number
 * @decimals 3
 * @default -0.06
 *
 * @param EdgeFadePx
 * @text 端フェード幅(px or 0-1)
 * @type number
 * @decimals 2
 * @min 0
 * @default 0.25
 */

(() => {
  "use strict";

  const pluginName = document.currentScript.src.split("/").pop().replace(/\.js$/i, "");
  const P = PluginManager.parameters(pluginName);

  const toBool = (v, d) => (v === "true" ? true : v === "false" ? false : d);
  const toNum  = (v, d) => {
    const n = Number(v);
    return Number.isFinite(n) ? n : d;
  };
  const clamp = (x, a, b) => Math.max(a, Math.min(b, x));
  const rand  = (a, b) => a + Math.random() * (b - a);
  const randi = (a, b) => Math.floor(rand(a, b + 1));

  const normalizeMinMax = (a, b) => (a <= b ? [a, b] : [b, a]);
  const normalizeMinMaxInt = (a, b) => {
    const aa = Math.floor(a);
    const bb = Math.floor(b);
    return aa <= bb ? [aa, bb] : [bb, aa];
  };

  const parseHexColor = (s) => {
    if (!s) return 0xffffff;
    const t = String(s).trim();
    if (t.startsWith("#")) return parseInt(t.slice(1), 16);
    if (t.startsWith("0x")) return parseInt(t.slice(2), 16);
    const n = parseInt(t, 16);
    return Number.isFinite(n) ? n : 0xffffff;
  };

  const parsePalette = (s) => {
    const raw = String(s || "").split(",").map(x => x.trim()).filter(Boolean);
    const list = raw.map(parseHexColor).filter(n => Number.isFinite(n));
    return list.length ? list : [0xffffff];
  };

  const jitterTint = (tint, jitter) => {
    if (!jitter || jitter <= 0) return tint;
    const r = (tint >> 16) & 255;
    const g = (tint >> 8) & 255;
    const b = tint & 255;
    const mult = 1 + rand(-jitter, jitter);
    const rr = clamp(Math.round(r * mult), 0, 255);
    const gg = clamp(Math.round(g * mult), 0, 255);
    const bb = clamp(Math.round(b * mult), 0, 255);
    return (rr << 16) | (gg << 8) | bb;
  };

  // px or ratio(0-1) を解像度に応じて解決
  const resolvePx = (value, base, minPx, maxPx) => {
    const v = Number(value);
    const px = (v > 0 && v <= 1) ? v * base : v;
    return clamp(Math.round(px), minPx, maxPx);
  };

  const params = {
    enabled: toBool(P.Enabled, true),
    layer: String(P.Layer || "aboveForeground"),
    count: clamp(toNum(P.ParticleCount, 80), 1, 300),

    startFilled: toBool(P.StartFilled, true),
    startPoolRatio: clamp(toNum(P.StartPoolRatio, 0.25), 0, 1),

    marginRaw: toNum(P.Margin, 160),
    poolHeightRaw: toNum(P.PoolHeight, 0.30),
    poolLeftWidthRate: clamp(toNum(P.PoolLeftWidthRate, 0.38), 0, 1),

    releaseMin: Math.max(1, Math.floor(toNum(P.ReleaseMinFrames, 120))),
    releaseMax: Math.max(1, Math.floor(toNum(P.ReleaseMaxFrames, 360))),
    releaseFade: Math.max(0, Math.floor(toNum(P.ReleaseFadeFrames, 40))),

    particleImage: String(P.ParticleImage || "").trim(),
    baseTextureSize: clamp(toNum(P.BaseTextureSize, 64), 16, 256),

    blendMode: String(P.BlendMode || "SCREEN"),

    sizeMin: toNum(P.SizeMin, 0.16),
    sizeMax: toNum(P.SizeMax, 0.52),
    alphaMin: toNum(P.AlphaMin, 0.18),
    alphaMax: toNum(P.AlphaMax, 0.60),

    palette: parsePalette(P.ColorPalette),
    colorJitter: clamp(toNum(P.ColorJitter, 0.10), 0, 0.5),

    driftXMin: toNum(P.DriftXMin, 0.06),
    driftXMax: toNum(P.DriftXMax, 0.22),

    riseYMin: toNum(P.RiseYMin, -0.55),
    riseYMax: toNum(P.RiseYMax, -0.18),

    poolBobAmpMin: toNum(P.PoolBobAmpMin, 0.6),
    poolBobAmpMax: toNum(P.PoolBobAmpMax, 2.8),

    swayAmpMin: toNum(P.SwayAmpMin, 0.6),
    swayAmpMax: toNum(P.SwayAmpMax, 3.4),
    swayFreqMin: toNum(P.SwayFreqMin, 0.010),
    swayFreqMax: toNum(P.SwayFreqMax, 0.035),

    rotMin: toNum(P.RotationSpeedMin, -0.010),
    rotMax: toNum(P.RotationSpeedMax, 0.010),

    windXMin: toNum(P.WindXMin, -0.06),
    windXMax: toNum(P.WindXMax, 0.10),
    windYMin: toNum(P.WindYMin, -0.04),
    windYMax: toNum(P.WindYMax, 0.02),
    windChangeFrames: Math.max(1, Math.floor(toNum(P.WindChangeFrames, 420))),

    gustRate: Math.max(0, toNum(P.GustChancePerSecond, 0.020)),
    gustDurMin: Math.max(1, Math.floor(toNum(P.GustDurationMin, 90))),
    gustDurMax: Math.max(1, Math.floor(toNum(P.GustDurationMax, 180))),
    gustXMin: toNum(P.GustXMin, 0.25),
    gustXMax: toNum(P.GustXMax, 0.55),
    gustYMin: toNum(P.GustYMin, -0.20),
    gustYMax: toNum(P.GustYMax, -0.06),

    edgeFadeRaw: toNum(P.EdgeFadePx, 0.25),
  };

  // Min/Max 正規化（逆転していても壊れないように）
  {
    [params.releaseMin, params.releaseMax] = normalizeMinMaxInt(params.releaseMin, params.releaseMax);
    [params.gustDurMin, params.gustDurMax] = normalizeMinMaxInt(params.gustDurMin, params.gustDurMax);

    [params.sizeMin, params.sizeMax] = normalizeMinMax(params.sizeMin, params.sizeMax);
    [params.alphaMin, params.alphaMax] = normalizeMinMax(params.alphaMin, params.alphaMax);

    [params.driftXMin, params.driftXMax] = normalizeMinMax(params.driftXMin, params.driftXMax);
    [params.riseYMin, params.riseYMax] = normalizeMinMax(params.riseYMin, params.riseYMax);

    [params.poolBobAmpMin, params.poolBobAmpMax] = normalizeMinMax(params.poolBobAmpMin, params.poolBobAmpMax);
    [params.swayAmpMin, params.swayAmpMax] = normalizeMinMax(params.swayAmpMin, params.swayAmpMax);
    [params.swayFreqMin, params.swayFreqMax] = normalizeMinMax(params.swayFreqMin, params.swayFreqMax);

    [params.rotMin, params.rotMax] = normalizeMinMax(params.rotMin, params.rotMax);

    [params.windXMin, params.windXMax] = normalizeMinMax(params.windXMin, params.windXMax);
    [params.windYMin, params.windYMax] = normalizeMinMax(params.windYMin, params.windYMax);

    [params.gustXMin, params.gustXMax] = normalizeMinMax(params.gustXMin, params.gustXMax);
    [params.gustYMin, params.gustYMax] = normalizeMinMax(params.gustYMin, params.gustYMax);
  }

  const resolveBlendMode = (modeStr) => {
    const m = String(modeStr || "").toUpperCase();
    if (m === "ADD") return PIXI.BLEND_MODES.ADD;
    if (m === "SCREEN") return PIXI.BLEND_MODES.SCREEN;
    return PIXI.BLEND_MODES.NORMAL;
  };

  const createDefaultLightBitmap = (size) => {
    const bmp = new Bitmap(size, size);
    const ctx = bmp.context;
    const cx = size * 0.5;
    const cy = size * 0.5;
    ctx.save();
    ctx.clearRect(0, 0, size, size);
    const g = ctx.createRadialGradient(cx, cy, 0, cx, cy, size * 0.5);
    g.addColorStop(0.00, "rgba(255,255,255,1.0)");
    g.addColorStop(0.25, "rgba(255,255,255,0.55)");
    g.addColorStop(0.55, "rgba(255,255,255,0.18)");
    g.addColorStop(1.00, "rgba(255,255,255,0.0)");
    ctx.fillStyle = g;
    ctx.fillRect(0, 0, size, size);
    ctx.restore();

    // 環境差対策：dirty反映ルートを増やす
    if (bmp._setDirty) bmp._setDirty();
    else if (bmp._baseTexture && bmp._baseTexture.update) bmp._baseTexture.update();

    return bmp;
  };

  const computeInsertIndex = (scene) => {
    const windowLayerIndex = scene._windowLayer ? scene.children.indexOf(scene._windowLayer) : -1;
    const titleSpriteIndex = scene._gameTitleSprite ? scene.children.indexOf(scene._gameTitleSprite) : -1;

    if (params.layer === "belowForeground" && titleSpriteIndex >= 0) return titleSpriteIndex;
    if (params.layer === "aboveForeground" && windowLayerIndex >= 0) return windowLayerIndex;
    if (params.layer === "aboveAll") return scene.children.length;
    return scene.children.length;
  };

  class TitleFantasyParticles {
    constructor(scene) {
      this._scene = scene;
      this._container = new PIXI.Container();
      this._blendMode = resolveBlendMode(params.blendMode);

      // ParticleImage が拡張子付きでも動くように：拡張子/パスを剥がす
      const pic = params.particleImage
        .replace(/^img\/pictures\//i, "")
        .replace(/\.(png|jpg|jpeg|webp)$/i, "");

      this._bitmap = pic ? ImageManager.loadPicture(pic) : createDefaultLightBitmap(params.baseTextureSize);

      // 常風
      this._windX = 0;
      this._windY = 0;
      this._windTargetX = rand(params.windXMin, params.windXMax);
      this._windTargetY = rand(params.windYMin, params.windYMax);
      this._windTimer = randi(Math.floor(params.windChangeFrames * 0.5), params.windChangeFrames);

      // 突風
      this._gustT = 0;
      this._gustX = 0;
      this._gustY = 0;
      this._gustTargetX = 0;
      this._gustTargetY = 0;

      this._time = rand(0, 9999);
      this._particles = [];
      this._createParticles(params.count);
    }

    container() { return this._container; }

    ensureAttached(scene) {
      if (!this._container) return;
      if (this._container.parent === scene) return;
      const idx = computeInsertIndex(scene);
      scene.addChildAt(this._container, clamp(idx, 0, scene.children.length));
    }

    destroy() {
      if (this._container) this._container.removeChildren();
      this._particles.length = 0;
      this._scene = null;
    }

    _runtime(w, h) {
      const base = Math.min(w, h);
      const m = resolvePx(params.marginRaw, base, 0, 4096);
      const poolH = resolvePx(params.poolHeightRaw, h, 0, 4096);
      const edge = resolvePx(params.edgeFadeRaw, base, 0, 4096);
      return { w, h, m, poolH, edge };
    }

    _pickTint() {
      const base = params.palette[Math.floor(Math.random() * params.palette.length)] || 0xffffff;
      return jitterTint(base, params.colorJitter);
    }

    // poolへ戻すときのX：左寄り“バイアス” + 一部均等（右側が薄くならないように）
    _poolSpawnX(w, m) {
      const span = w + m * 2;
      const r = clamp(params.poolLeftWidthRate, 0, 1);

      // r=0 → 強く左寄り / r=1 → ほぼ均等
      const pow = 1 + (1 - r) * 2.2; // 1.0 .. 3.2

      // 右側の欠けを確実に埋めるため、一定割合は均等分布にする
      const UNIFORM_MIX = 0.30; // 0.0..1.0（大きいほど均等）
      const u = Math.random();
      const uu = (Math.random() < UNIFORM_MIX) ? u : Math.pow(u, pow);

      return uu * span - m;
    }

    _createParticles(n) {
      const w = Graphics.width;
      const h = Graphics.height;
      const { m, poolH } = this._runtime(w, h);

      for (let i = 0; i < n; i++) {
        const sp = new Sprite(this._bitmap);
        sp.anchor.set(0.5, 0.5);
        sp.blendMode = this._blendMode;
        sp.tint = this._pickTint();

        const p = {
          sprite: sp,
          x: 0, y: 0,
          vx: 0, vy: 0,
          omega: 0,
          baseAlpha: 0,
          baseScale: 1,

          state: "rise",      // pool / rise
          hold: 0,
          releaseAge: 0,

          poolTargetY: 0,

          swayAmp: 0,
          swayFreq: 0,
          swayPhase: rand(0, Math.PI * 2),

          poolBobAmp: 0,
          poolBobFreq: rand(0.010, 0.030),
          poolBobPhase: rand(0, Math.PI * 2),

          flickerAmp: rand(0.04, 0.14),
          flickerFreq: rand(0.025, 0.070),
          breathFreq: rand(0.010, 0.028),
          seed: rand(0, 9999),
        };

        // 起動直後から“既に居る”見え方にする初期化
        if (params.startFilled) {
          if (Math.random() < params.startPoolRatio) {
            this._enterPool(p, /*initial*/true);
            // pool粒子は最初から底に居る（落ちて集まる過程を作らない）
            p.x = this._poolSpawnX(w, m);
            p.poolTargetY = rand(h - poolH, h + m * 0.15);
            p.y = p.poolTargetY + rand(-12, 12);
            // 待ち時間もばらけさせる（すぐ何体か上がる）
            p.hold = rand(0, params.releaseMax);
          } else {
            this._enterRise(p, /*initial*/true);
            p.x = rand(-m, w + m);
            p.y = rand(-m, h + m);
            // 起動時に“ふわっと出現待ち”を作らない
            p.releaseAge = params.releaseFade;
          }
        } else {
          this._enterPool(p, /*initial*/false);
        }

        this._particles.push(p);
        this._container.addChild(sp);
      }
    }

    _commonStyle(p) {
      p.baseAlpha = rand(params.alphaMin, params.alphaMax);
      p.baseScale = rand(params.sizeMin, params.sizeMax);
      p.omega = rand(params.rotMin, params.rotMax);
      p.sprite.tint = this._pickTint();
      p.sprite.rotation = rand(0, Math.PI * 2);
      p.sprite.scale.set(p.baseScale);
    }

    _enterPool(p, initial) {
      const w = Graphics.width;
      const h = Graphics.height;
      const { m, poolH } = this._runtime(w, h);

      p.state = "pool";
      p.releaseAge = 0;

      // pool中は弱い右流れ（溜まりはできるが“静か”）
      p.vx = rand(params.driftXMin, params.driftXMax) * 0.25;
      p.vy = 0;

      p.swayAmp = rand(params.swayAmpMin, params.swayAmpMax) * 0.55;
      p.swayFreq = rand(params.swayFreqMin, params.swayFreqMax) * 0.70;
      p.poolBobAmp = rand(params.poolBobAmpMin, params.poolBobAmpMax);

      p.hold = randi(params.releaseMin, params.releaseMax);
      p.poolTargetY = rand(h - poolH, h + m * 0.15);

      // 初期以外は pool へ戻す（Xは左寄りバイアス+均等mix）
      if (!initial) {
        p.x = this._poolSpawnX(w, m);
        p.y = rand(h - poolH, h + m);
      }

      this._commonStyle(p);
      p.sprite.alpha = p.baseAlpha * 0.65;
    }

    _enterRise(p, initial) {
      p.state = "rise";
      p.releaseAge = initial ? params.releaseFade : 0;

      p.vx = rand(params.driftXMin, params.driftXMax);
      const vy = rand(params.riseYMin, params.riseYMax);
      p.vy = -Math.abs(vy); // 上方向固定（“静かな幻想”の意図）

      p.swayAmp = rand(params.swayAmpMin, params.swayAmpMax);
      p.swayFreq = rand(params.swayFreqMin, params.swayFreqMax);

      this._commonStyle(p);
      p.sprite.alpha = p.baseAlpha;
    }

    _updateWindAndGust(dt) {
      // 常風ターゲット更新
      this._windTimer -= dt;
      if (this._windTimer <= 0) {
        this._windTargetX = rand(params.windXMin, params.windXMax);
        this._windTargetY = rand(params.windYMin, params.windYMax);
        this._windTimer = randi(Math.floor(params.windChangeFrames * 0.5), params.windChangeFrames);
      }
      // 追従
      const k = 0.010 * dt;
      this._windX += (this._windTargetX - this._windX) * k;
      this._windY += (this._windTargetY - this._windY) * k;

      // 突風発生(秒レート近似)
      const pGust = (params.gustRate / 60) * dt;
      if (this._gustT <= 0 && pGust > 0 && Math.random() < pGust) {
        this._gustT = randi(params.gustDurMin, params.gustDurMax);
        this._gustTargetX = rand(params.gustXMin, params.gustXMax);
        this._gustTargetY = -Math.abs(rand(params.gustYMin, params.gustYMax)); // 上方向寄与
      }

      if (this._gustT > 0) {
        this._gustT -= dt;
        const gk = 0.030 * dt;
        this._gustX += (this._gustTargetX - this._gustX) * gk;
        this._gustY += (this._gustTargetY - this._gustY) * gk;
        if (this._gustT <= 0) {
          this._gustTargetX = 0;
          this._gustTargetY = 0;
        }
      } else {
        const back = 0.020 * dt;
        this._gustX += (0 - this._gustX) * back;
        this._gustY += (0 - this._gustY) * back;
      }
    }

    _edgeFade(x, y, w, h, m, edge) {
      if (edge <= 0) return 1.0;
      const left = clamp((x + m) / edge, 0, 1);
      const right = clamp((w + m - x) / edge, 0, 1);
      const top = clamp((y + m) / edge, 0, 1);
      const bottom = clamp((h + m - y) / edge, 0, 1);
      return Math.min(left, right, top, bottom);
    }

    update() {
      const dt = Graphics.app && Graphics.app.ticker ? (Graphics.app.ticker.deltaMS / 16.6667) : 1;

      const w = Graphics.width;
      const h = Graphics.height;
      const { m, poolH, edge } = this._runtime(w, h);

      this._time += dt;
      this._updateWindAndGust(dt);

      const windX = this._windX + this._gustX;
      const windY = this._windY + this._gustY;

      for (const p of this._particles) {
        if (p.state === "pool") {
          p.hold -= dt;

          p.x += (p.vx + windX * 0.30) * dt;

          // poolTargetY へ静かに収束 + bob（“溜まり”を安定させる）
          const bob = Math.sin(p.poolBobPhase + (this._time + p.seed) * p.poolBobFreq) * p.poolBobAmp;
          const target = p.poolTargetY + bob;
          p.y += (target - p.y) * (0.040 * dt);

          // pool帯が画面外へ消えない程度に回り込み
          if (p.x < -m) p.x = w + m;
          if (p.x > w + m) p.x = -m;

          if (p.hold <= 0) {
            // pool→rise：その場からふわっと抜ける
            this._enterRise(p, /*initial*/false);
          }
        } else {
          p.releaseAge += dt;

          const sway = Math.sin(p.swayPhase + (this._time + p.seed) * p.swayFreq) * p.swayAmp;
          p.x += (p.vx + windX) * dt + sway * 0.10;
          p.y += (p.vy + windY) * dt;

          // 左に押し戻される突風対策
          if (p.x < -m) p.x = w + m;

          // 右へ抜けたら「回り込み」
          if (p.x > w + m) p.x = -m;

          // 上へ抜けたら pool へ戻す（永久ループ）
          if (p.y < -m) {
            this._enterPool(p, /*initial*/false);
          }
        }

        // 表示更新
        p.sprite.rotation += p.omega * dt;

        const edgeMul = this._edgeFade(p.x, p.y, w, h, m, edge);

        let releaseMul = 1.0;
        if (p.state === "rise" && params.releaseFade > 0) {
          releaseMul = clamp(p.releaseAge / params.releaseFade, 0, 1);
        }

        const flicker = 1.0 + p.flickerAmp * Math.sin((this._time + p.seed) * p.flickerFreq);
        const breathe = 1.0 + 0.10 * Math.sin((this._time + p.seed) * p.breathFreq);
        const stateMul = (p.state === "pool") ? 0.65 : 1.00;

        p.sprite.alpha = clamp(p.baseAlpha * stateMul * edgeMul * releaseMul * flicker, 0, 1);
        p.sprite.scale.set(p.baseScale * breathe);
        p.sprite.x = p.x;
        p.sprite.y = p.y;
      }
    }
  }

  const installToTitleScene = () => {
    const _create = Scene_Title.prototype.create;
    Scene_Title.prototype.create = function() {
      _create.call(this);
      if (!params.enabled) return;

      if (!this._titleFantasyParticles) {
        this._titleFantasyParticles = new TitleFantasyParticles(this);
      }
      this._titleFantasyParticles.ensureAttached(this);
    };

    const _update = Scene_Title.prototype.update;
    Scene_Title.prototype.update = function() {
      _update.call(this);
      if (this._titleFantasyParticles) {
        this._titleFantasyParticles.ensureAttached(this);
        this._titleFantasyParticles.update();
      }
    };

    const _terminate = Scene_Title.prototype.terminate;
    Scene_Title.prototype.terminate = function() {
      if (this._titleFantasyParticles) {
        const c = this._titleFantasyParticles.container();
        if (c && c.parent) c.parent.removeChild(c);
        this._titleFantasyParticles.destroy();
        this._titleFantasyParticles = null;
      }
      _terminate.call(this);
    };
  };

  if (typeof Scene_Title !== "undefined") installToTitleScene();
})();

